Rem ImageCatalog.vbs
Rem An InDesign VBScript
Rem
Rem Creates an image catalog from the graphic files in a selected folder.
Rem Each file can be labeled with the file name, and the labels are placed on
Rem a separate layer and formatted using a paragraph style ("label") you can
Rem modify to change the appearance of the labels.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem Or visit the InDesign Scripting User to User forum at http://www.adobeforums.com .
Rem
Rem The myExtensions array contains the extensions of the graphic file types you want
Rem to include in the catalog. You can remove extensions from or add extensions to this list.
myExtensions = Array(".jpg", ".jpeg", ".eps", ".ps", ".pdf", ".tif", ".tiff", ".gif", ".psd", ".ai")
main
Function main()
	ReDim myFileList(0)
	Set myInDesign = CreateObject("InDesign.Application.2022")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll	
	Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
	Rem Creating a folder browser in VBScript can be a problem (relying on either Windows API calls
	Rem or use of ActiveX controls which may not be present on a given system). Instead, we'll use
	Rem InDesign's built-in JavaScript to display a file browser. DoScript can return a value;
	Rem in this example, it's the name of the folder.
	myJavaScript = "myFolder = Folder.selectDialog(""Choose a Folder""); myFolderName = myFolder.fsName;"
	Rem The following line shows how to return a value from a script called with the DoScript method.
	myFolderName = myInDesign.doScript(myJavaScript, idScriptLanguage.idJavascript)
	If myFileSystemObject.FolderExists(myFolderName) Then
		Set myFolder = myFileSystemObject.GetFolder(myFolderName)
		For Each File In myFolder.Files
			myFilePath = File.Path
			If myCheckFileType(myFilePath, myExtensions) = True Then
				Rem Add the file to the list of files to be placed.
				If Not (IsEmpty(myFileList(0))) Then
					ReDim Preserve myFileList(UBound(myFileList) + 1)
				End If
				myFileList(UBound(myFileList)) = myFilePath
			End If
		Next
		If Not (IsEmpty(myFileList(0))) Then
			myDisplayDialog myInDesign, myFileList, myFolderName
		Else
			MsgBox "No files found to place in the specified folder."
		End If
	End If
End Function
Function myDisplayDialog(myInDesign, myFileList, myFolderName)
    myLabelWidth = 130
    Set myDialog = myInDesign.dialogs.Add
    myDialog.Name = "Image Catalog"
    myStyleNames = myGetStyleNames(myInDesign)
    myLabelTypes = Array("File Name", "File Path", "XMP Description", "XMP Author")
    myLayerNames = Array("Layer 1", "Labels")
    With myDialog.DialogColumns.Add
        With .DialogRows.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .staticLabel = "Information"
                End With
            End With
        End With
        With .BorderPanels.Add
            With .DialogColumns.Add
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Source Folder:"
                        .MinWidth = myLabelWidth
                    End With
                    With .StaticTexts.Add
                        .staticLabel = myFolderName
                    End With
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Images:"
                        .MinWidth = myLabelWidth
                    End With
                    With .StaticTexts.Add
                        .staticLabel = CStr(UBound(myFileList)+1)
                    End With
                End With
            End With
        End With
        With .DialogRows.Add
            With .StaticTexts.Add
                .staticLabel = "Options"
            End With
        End With
        With .BorderPanels.Add
            With .DialogColumns.Add
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Rows:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myNumberOfRowsField = .integerEditboxes.Add
                    myNumberOfRowsField.editValue = 3
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Number of Columns:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myNumberOfColumnsField = .integerEditboxes.Add
                    myNumberOfColumnsField.editValue = 3
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Horizontal Offset:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myHorizontalOffsetField = .measurementEditboxes.Add
                    myHorizontalOffsetField.editValue = 12
                    myHorizontalOffsetField.EditUnits = idMeasurementUnits.idPoints
                End With
                With .DialogRows.Add
                    With .StaticTexts.Add
                        .staticLabel = "Vertical Offset:"
                        .MinWidth = myLabelWidth
                    End With
                    Set myVerticalOffsetField = .measurementEditboxes.Add
                    myVerticalOffsetField.editValue = 12
                    myVerticalOffsetField.EditUnits = idMeasurementUnits.idPoints
                End With
                With .DialogRows.Add
                    With .DialogColumns.Add
                        With .StaticTexts.Add
                            .staticLabel = "Fitting:"
                            .MinWidth = myLabelWidth
                        End With
                    End With
                    With .DialogColumns.Add
                        Set myFitProportionalCheckbox = .checkboxControls.Add
                        myFitProportionalCheckbox.staticLabel = "Proportional"
                        myFitProportionalCheckbox.checkedState = True
                        Set myFitCenterContentCheckbox = .checkboxControls.Add
                        myFitCenterContentCheckbox.staticLabel = "Center Content"
                        myFitCenterContentCheckbox.checkedState = True
                        Set myFitFrameToContentCheckbox = .checkboxControls.Add
                        myFitFrameToContentCheckbox.staticLabel = "Frame to Content"
                        myFitFrameToContentCheckbox.checkedState = True
                    End With
                End With
                With .DialogRows.Add
                    Set myRemoveEmptyFramesCheckbox = .checkboxControls.Add
                    myRemoveEmptyFramesCheckbox.staticLabel = "Remove Empty Frames"
                    myRemoveEmptyFramesCheckbox.checkedState = True
                End With
			End With
		End With
		Rem "Spacer" line to separate Labels group from the border.
		With .StaticTexts.Add
			.StaticLabel = ""
		End With
        Set myLabelsGroup = .EnablingGroups.Add
        With myLabelsGroup
        	.StaticLabel = "Labels"
        	.CheckedState = True
        	With .DialogColumns.Add
        		Rem Label type dropdown and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Type:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelTypeDropdown = .Dropdowns.Add
        				With myLabelTypeDropdown
        					.StringList = myLabelTypes
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        		Rem Label text frame height and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Height:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelHeightField = .MeasurementEditboxes.Add
        				With myLabelHeightField
        					.EditValue = 24
        					.EditUnits = idMeasurementUnits.idPoints
        				End With
        			End With
        		End With
        		Rem Label text frame offset and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Offset:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelOffsetField = .MeasurementEditboxes.Add
        				With myLabelOffsetField
        					.EditValue = 0
        					.EditUnits = idMeasurementUnits.idPoints
        				End With
        			End With
        		End With
        		Rem Label paragraph style and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Style:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelStyleDropdown = .Dropdowns.Add
        				With myLabelStyleDropdown
        					.StringList = myStyleNames
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        		Rem Label layer and associated static text.
        		With .DialogRows.Add
        			With .DialogColumns.Add
        				With .StaticTexts.Add
        					.StaticLabel = "Label Layer:"
        					.MinWidth = myLabelWidth
        				End With
        			End With
        			With .DialogColumns.Add
        				Set myLabelLayerDropdown = .Dropdowns.Add
        				With myLabelLayerDropdown
        					.StringList = myLayerNames
        					.SelectedIndex = 0
        				End With
        			End With
        		End With
        	End With
		End With
    End With
    myResult = myDialog.Show
    If myResult = True Then
        myNumberOfRows = myNumberOfRowsField.editValue
        myNumberOfColumns = myNumberOfColumnsField.editValue
        myRemoveEmptyFrames = myRemoveEmptyFramesCheckbox.checkedState
        myFitProportional = myFitProportionalCheckbox.checkedState
        myFitCenterContent = myFitCenterContentCheckbox.checkedState
        myFitFrameToContent = myFitFrameToContentCheckbox.checkedState
        myHorizontalOffset = myHorizontalOffsetField.editValue
        myVerticalOffset = myVerticalOffsetField.editValue
        Rem Label options.
        myMakeLabels = myLabelsGroup.CheckedState
        myLabelHeight = myLabelHeightField.EditValue
        myLabelOffset = myLabelOffsetField.EditValue
        myLabelType = myLabelTypes(myLabelTypeDropdown.SelectedIndex)
        myLabelStyleName = myStyleNames(myLabelStyleDropdown.SelectedIndex)
        myLabelLayerName = myLayerNames(myLabelLayerDropdown.SelectedIndex)
        myDialog.destroy
        myMakeImageCatalog myInDesign, myFileList, myNumberOfRows, myNumberOfColumns, myRemoveEmptyFrames, myFitProportional, myFitCenterContent, myFitFrameToContent, myHorizontalOffset, myVerticalOffset, myMakeLabels, myLabelHeight, myLabelOffset, myLabelType, myLabelStyleName, myLabelLayerName
        MsgBox "Done!"
    Else
        myDialog.destroy
    End If
End Function
Function myMakeImageCatalog(myInDesign, myFileList, myNumberOfRows, myNumberOfColumns, myRemoveEmptyFrames, myFitProportional, myFitCenterContent, myFitFrameToContent, myHorizontalOffset, myVerticalOffset, myMakeLabels, myLabelHeight, myLabelOffset, myLabelType, myLabelStyleName, myLabelLayerName)
    Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
    Set myDocument = myInDesign.documents.Add
    myDocument.viewPreferences.horizontalMeasurementUnits = idMeasurementUnits.idPoints
    myDocument.viewPreferences.verticalMeasurementUnits = idMeasurementUnits.idPoints
    Set myDocumentPreferences = myDocument.documentPreferences
    myFramesPerPage = myNumberOfRows * myNumberOfColumns
    myNumberOfFrames = UBound(myFileList)+1
    myNumberOfPages = Round(myNumberOfFrames / myFramesPerPage)
    If (myNumberOfPages * myFramesPerPage) < myNumberOfFrames Then
        myNumberOfPages = myNumberOfPages + 1
    End If
    myDocumentPreferences.pagesPerDocument = CInt(myNumberOfPages)
    myDocumentPreferences.facingPages = False
    Set myPage = myDocument.Pages.Item(1)
    Set myMarginPreferences = myPage.marginPreferences
    myLeftMargin = myMarginPreferences.Left
    myTopMargin = myMarginPreferences.Top
    myRightMargin = myMarginPreferences.Right
    myBottomMargin = myMarginPreferences.bottom
    myLiveWidth = (myDocumentPreferences.pageWidth - (myLeftMargin + myRightMargin)) + myHorizontalOffset
    myLiveHeight = myDocumentPreferences.pageHeight - (myTopMargin + myBottomMargin)
    myColumnWidth = myLiveWidth / myNumberOfColumns
    myFrameWidth = myColumnWidth - myHorizontalOffset
    myRowHeight = myLiveHeight / myNumberOfRows
    myFrameHeight = myRowHeight - myVerticalOffset
    Set myPages = myDocument.Pages
    If myMakeLabels = True Then
        Rem Create the label paragraph style if it does not already exist.
        Err.Clear
        On Error Resume Next
        Set myLabelStyle = myDocument.paragraphStyles.Item(myLabelStyleName)
        If Err.Number <> 0 Then
            Set myLabelStyle = myDocument.paragraphStyles.Add
            myLabelStyle.Name = myLabelStyleName
            Err.Clear
        End If
        On Error GoTo 0
        Rem Create the label layer if it does not already exist.
        Err.Clear
        On Error Resume Next
        Set myLabelLayer = myDocument.layers.Item(myLabelLayerName)
        If Err.Number <> 0 Then
            Set myLabelLayer = myDocument.layers.Add
            myLabelLayer.Name = myLabelLayerName
            Err.Clear
        End If
        On Error GoTo 0
    End If
    Rem Construct the frames in reverse order. Don't laugh--this will
    Rem save us time later (when we place the graphics).
    For myCounter = myDocument.Pages.Count To 1 Step -1
        Set myPage = myPages.Item(CInt(myCounter))
        For myRowCounter = myNumberOfRows To 1 Step -1
            myY1 = myTopMargin + (myRowHeight * (myRowCounter - 1))
            myY2 = myY1 + myFrameHeight
            For myColumnCounter = myNumberOfColumns To 1 Step -1
                myX1 = myLeftMargin + (myColumnWidth * (myColumnCounter - 1))
                myX2 = myX1 + myFrameWidth
                Set myRectangle = myPage.Rectangles.Add
                myRectangle.GeometricBounds = Array(myY1, myX1, myY2, myX2)
                myRectangle.strokeWeight = 0
                myRectangle.strokeColor = myDocument.Swatches.Item("None")
            Next
        Next
    Next
    Rem Because we constructed the frames in reverse order, rectangle 1
    Rem is the first rectangle on page 1, so we can simply iterate through
    Rem the rectangles, placing a file in each one in turn. myFiles = myFolder.Files;
    For myCounter = 1 To myNumberOfFrames
        myFile = myFileList(myCounter-1)
        Set myRectangle = myDocument.Rectangles.Item(myCounter)
        myRectangle.place myFile
        myRectangle.Label
        Rem Apply fitting options as specified.
        If myFitProportional Then
            myRectangle.fit idFitOptions.idProportionally
        End If
        If myFitCenterContent Then
            myRectangle.fit idFitOptions.idCenterContent
        End If
        If myFitFrameToContent Then
            myRectangle.fit idFitOptions.idFrameToContent
        End If
        Rem Add the label, if necessary.
        If myMakeLabels = True Then
        	Set myRegExp = New RegExp
			myRegExp.Pattern = "^\s*$"
			myRegExp.IgnoreCase = True
			myX1 = myRectangle.GeometricBounds(1)
			myY1 = myRectangle.GeometricBounds(2) + myLabelOffset
			myX2 = myRectangle.GeometricBounds(3)
			myY2 = myY1 + myLabelHeight
			Select Case myLabelType
				Case "File Name":
					myString = myFileSystemObject.GetFileName(myFile)
				Case "File Path":
					myString = myFileSystemObject.GetAbsolutePathName(myFile)
				Case "XMP Author":
					On Error Resume Next
						myString = myLink.LinkXmp.Author
						If myRegExp.Replace(myString, "") = "" Then
							Error
						End If
					If Err.Number <> 0 Then
						myString = "No author available."
						Err.Clear
					End If
					On Error Goto 0
				Case "XMP Description":
					On Error Resume Next
						myString = myLink.LinkXmp.Description
						If myRegExp.Replace(myString, "") = "" Then
							Error
						End If
					If Err.Number <> 0 Then
						myString = "No description available."
						Err.Clear
					End If
					On Error Goto 0
			End Select
			Set myTextFrame = myRectangle.Parent.textFrames.Add
			myTextFrame.ItemLayer = myLabelLayer
			myTextFrame.GeometricBounds = Array(myY1, myX1, myY2, myX2)
			myTextFrame.Contents = myString
			myTextFrame.textFramePreferences.firstBaselineOffset = idFirstBaseline.idLeadingOffset
			myTextFrame.ParentStory.Texts.Item(1).appliedParagraphStyle = myLabelStyle
        End If
    Next
    If myRemoveEmptyFrames = True Then
		For myCounter = myDocument.Rectangles.Count To 1 Step -1
			If myDocument.Rectangles.Item(myCounter).contentType = idContentType.idUnassigned Then
				myDocument.Rectangles.Item(myCounter).Delete
			Else
				Rem As soon as you encounter a rectangle with content, exit the loop.
				Exit For
			End If
		Next
    End If
End Function
Function myCheckFileType(myFilePath, myExtensions)
	myFilePath = LCase(myFilePath)
    For Each myExtension In myExtensions
        If InStr(myFilePath, myExtension) <> 0 Then
            myCheckFileType = True
            Exit Function
        End If
    Next
    myCheckFileType = False
End Function
Rem Return an array of paragraph style names.
Function myGetStyleNames(myInDesign)
	myAddLabelStyle = True
	ReDim myStyleNames(0)
	For myCounter = 1 To myInDesign.ParagraphStyles.Count
		If Not (IsEmpty(myStyleNames(0))) Then
	    	ReDim Preserve myStyleNames(UBound(myStyleNames) + 1)
	    End If
		myStyleNames(UBound(myStyleNames)) = myInDesign.ParagraphStyles.Item(myCounter).Name
		If myInDesign.ParagraphStyles.Item(myCounter).Name = "Labels" Then
			myAddLabelStyle = False
		End If
	Next
	If myAddLabelStyle = True Then
		ReDim Preserve myStyleNames(UBound(myStyleNames)+1)
		myStyleNames(UBound(myStyleNames)) = "Labels"		
	End If
	myGetStyleNames = myStyleNames
End Function
'' SIG '' Begin signature block
'' SIG '' MIIhPQYJKoZIhvcNAQcCoIIhLjCCISoCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' KfymTfw02wSzLtAYE2H88he2tizKoaNlq+oUIpNAGKWg
'' SIG '' ggyUMIIF0DCCBLigAwIBAgIQCHK5i9aO5k82w+FiDw5C
'' SIG '' dzANBgkqhkiG9w0BAQsFADBsMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSswKQYDVQQDEyJEaWdpQ2Vy
'' SIG '' dCBFViBDb2RlIFNpZ25pbmcgQ0EgKFNIQTIpMB4XDTIw
'' SIG '' MTIxOTAwMDAwMFoXDTIyMTIyMTIzNTk1OVowgeYxEzAR
'' SIG '' BgsrBgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIB
'' SIG '' AhMIRGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3Jn
'' SIG '' YW5pemF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYD
'' SIG '' VQQGEwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNh
'' SIG '' biBKb3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYD
'' SIG '' VQQLEyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29w
'' SIG '' eSwgTXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCASIw
'' SIG '' DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKraHbKp
'' SIG '' 7LWhQb1eUijAwzznOPnL0ar6Bh5IyTX0X3Nmx0y/zOPp
'' SIG '' A769ixYCLdufIms+U9xBOpXey6jajunMtomUtqLHkYMN
'' SIG '' DT2jzWOCqLvnwrHjehFxLYwYIt9ymU0VqJHPcIO+0lfA
'' SIG '' s4ZoskzD5OepJsCVjURfwILDS/l51c42dKXqbPmr25vz
'' SIG '' UQBlKIg7JV7JJGj29DQMOeCI9LSY4dAUYsfaEfDx0dcb
'' SIG '' T/oDlUZAH/mGrvFonoqGwLDNNSpWkFauC3ryUlgMJ6if
'' SIG '' JX9iZIrdm6Glo/EvxbsHMzsL2TclgBXoa0mYq4PYgMH1
'' SIG '' 5XYNA9y4x7QNLRmWPpXU+4gnfN0CAwEAAaOCAfEwggHt
'' SIG '' MB8GA1UdIwQYMBaAFI/ofvBtMmoABSPHcJdqOpD/a+rU
'' SIG '' MB0GA1UdDgQWBBS5nvze3WzR/65rffm/COInbJUTtzAu
'' SIG '' BgNVHREEJzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVM
'' SIG '' QVdBUkUtMjc0ODEyOTAOBgNVHQ8BAf8EBAMCB4AwEwYD
'' SIG '' VR0lBAwwCgYIKwYBBQUHAwMwewYDVR0fBHQwcjA3oDWg
'' SIG '' M4YxaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0VWQ29k
'' SIG '' ZVNpZ25pbmdTSEEyLWcxLmNybDA3oDWgM4YxaHR0cDov
'' SIG '' L2NybDQuZGlnaWNlcnQuY29tL0VWQ29kZVNpZ25pbmdT
'' SIG '' SEEyLWcxLmNybDBLBgNVHSAERDBCMDcGCWCGSAGG/WwD
'' SIG '' AjAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
'' SIG '' Y2VydC5jb20vQ1BTMAcGBWeBDAEDMH4GCCsGAQUFBwEB
'' SIG '' BHIwcDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
'' SIG '' aWNlcnQuY29tMEgGCCsGAQUFBzAChjxodHRwOi8vY2Fj
'' SIG '' ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRFVkNvZGVT
'' SIG '' aWduaW5nQ0EtU0hBMi5jcnQwDAYDVR0TAQH/BAIwADAN
'' SIG '' BgkqhkiG9w0BAQsFAAOCAQEALl6DDJXB+ruS0J/n4zd6
'' SIG '' IxqKGcJnnn72tgm50zJjxjqD9f4gMlY5lFA4Dcz9D6ur
'' SIG '' Yc734LqeT9p+oUv2PI4KdFMpSmhdrKLkI8eTK46NHVW8
'' SIG '' U/GZctdsVZR1JvtI1ByrH/DkQS7HTKWwtbyOkKU7lfYK
'' SIG '' HIS+qUso64cM6M2asUym8CdN/3FbwJ9dGJDX9yfMISbX
'' SIG '' oIDfInSnVDD+Z9EnnpsEw9obZwNHaGdd1NJRuPrlwatk
'' SIG '' YdBJuo5kN5/eBMYb95FbK8zZf8e+5FTnEJ0PmxRakjlF
'' SIG '' xh5Zg3CCjiIjr3w/uUkOaa2PjpkNy/vuUYxCuJb53Ca6
'' SIG '' i+n/WMWWjYJLNDCCBrwwggWkoAMCAQICEAPxtOFfOoLx
'' SIG '' FJZ4s9fYR1wwDQYJKoZIhvcNAQELBQAwbDELMAkGA1UE
'' SIG '' BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
'' SIG '' A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTErMCkGA1UEAxMi
'' SIG '' RGlnaUNlcnQgSGlnaCBBc3N1cmFuY2UgRVYgUm9vdCBD
'' SIG '' QTAeFw0xMjA0MTgxMjAwMDBaFw0yNzA0MTgxMjAwMDBa
'' SIG '' MGwxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
'' SIG '' dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20x
'' SIG '' KzApBgNVBAMTIkRpZ2lDZXJ0IEVWIENvZGUgU2lnbmlu
'' SIG '' ZyBDQSAoU0hBMikwggEiMA0GCSqGSIb3DQEBAQUAA4IB
'' SIG '' DwAwggEKAoIBAQCnU/oPsrUT8WTPhID8roA10bbXx6Ms
'' SIG '' rBosrPGErDo1EjqSkbpX5MTJ8y+oSDy31m7clyK6UXlh
'' SIG '' r0MvDbebtEkxrkRYPqShlqeHTyN+w2xlJJBVPqHKI3zF
'' SIG '' QunEemJFm33eY3TLnmMl+ISamq1FT659H8gTy3WbyeHh
'' SIG '' ivgLDJj0yj7QRap6HqVYkzY0visuKzFYZrQyEJ+d8FKh
'' SIG '' 7+g+03byQFrc+mo9G0utdrCMXO42uoPqMKhM3vELKlhB
'' SIG '' iK4AiasD0RaCICJ2615UOBJi4dJwJNvtH3DSZAmALeK2
'' SIG '' nc4f8rsh82zb2LMZe4pQn+/sNgpcmrdK0wigOXn93b89
'' SIG '' OgklAgMBAAGjggNYMIIDVDASBgNVHRMBAf8ECDAGAQH/
'' SIG '' AgEAMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggr
'' SIG '' BgEFBQcDAzB/BggrBgEFBQcBAQRzMHEwJAYIKwYBBQUH
'' SIG '' MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBJBggr
'' SIG '' BgEFBQcwAoY9aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0SGlnaEFzc3VyYW5jZUVWUm9vdENB
'' SIG '' LmNydDCBjwYDVR0fBIGHMIGEMECgPqA8hjpodHRwOi8v
'' SIG '' Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRIaWdoQXNz
'' SIG '' dXJhbmNlRVZSb290Q0EuY3JsMECgPqA8hjpodHRwOi8v
'' SIG '' Y3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRIaWdoQXNz
'' SIG '' dXJhbmNlRVZSb290Q0EuY3JsMIIBxAYDVR0gBIIBuzCC
'' SIG '' AbcwggGzBglghkgBhv1sAwIwggGkMDoGCCsGAQUFBwIB
'' SIG '' Fi5odHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9zc2wtY3Bz
'' SIG '' LXJlcG9zaXRvcnkuaHRtMIIBZAYIKwYBBQUHAgIwggFW
'' SIG '' HoIBUgBBAG4AeQAgAHUAcwBlACAAbwBmACAAdABoAGkA
'' SIG '' cwAgAEMAZQByAHQAaQBmAGkAYwBhAHQAZQAgAGMAbwBu
'' SIG '' AHMAdABpAHQAdQB0AGUAcwAgAGEAYwBjAGUAcAB0AGEA
'' SIG '' bgBjAGUAIABvAGYAIAB0AGgAZQAgAEQAaQBnAGkAQwBl
'' SIG '' AHIAdAAgAEMAUAAvAEMAUABTACAAYQBuAGQAIAB0AGgA
'' SIG '' ZQAgAFIAZQBsAHkAaQBuAGcAIABQAGEAcgB0AHkAIABB
'' SIG '' AGcAcgBlAGUAbQBlAG4AdAAgAHcAaABpAGMAaAAgAGwA
'' SIG '' aQBtAGkAdAAgAGwAaQBhAGIAaQBsAGkAdAB5ACAAYQBu
'' SIG '' AGQAIABhAHIAZQAgAGkAbgBjAG8AcgBwAG8AcgBhAHQA
'' SIG '' ZQBkACAAaABlAHIAZQBpAG4AIABiAHkAIAByAGUAZgBl
'' SIG '' AHIAZQBuAGMAZQAuMB0GA1UdDgQWBBSP6H7wbTJqAAUj
'' SIG '' x3CXajqQ/2vq1DAfBgNVHSMEGDAWgBSxPsNpA/i/RwHU
'' SIG '' mCYaCALvY2QrwzANBgkqhkiG9w0BAQsFAAOCAQEAGTNK
'' SIG '' DIEzN9utNsnkyTq7tRsueqLi9ENCF56/TqFN4bHb6YHd
'' SIG '' nwHy5IjV6f4J/SHB7F2A0vDWwUPC/ncr2/nXkTPObNWy
'' SIG '' GTvmLtbJk0+IQI7N4fV+8Q/GWVZy6OtqQb0c1UbVfEnK
'' SIG '' ZjgVwb/gkXB3h9zJjTHJDCmiM+2N4ofNiY0/G//V4BqX
'' SIG '' i3zabfuoxrI6Zmt7AbPN2KY07BIBq5VYpcRTV6hg5ucC
'' SIG '' EqC5I2SiTbt8gSVkIb7P7kIYQ5e7pTcGr03/JqVNYUvs
'' SIG '' RkG4Zc64eZ4IlguBjIo7j8eZjKMqbphtXmHGlreKuWEt
'' SIG '' k7jrDgRD1/X+pvBi1JlqpcHB8GSUgDGCFAEwghP9AgEB
'' SIG '' MIGAMGwxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
'' SIG '' Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5j
'' SIG '' b20xKzApBgNVBAMTIkRpZ2lDZXJ0IEVWIENvZGUgU2ln
'' SIG '' bmluZyBDQSAoU0hBMikCEAhyuYvWjuZPNsPhYg8OQncw
'' SIG '' DQYJYIZIAWUDBAIBBQCggZwwGQYJKoZIhvcNAQkDMQwG
'' SIG '' CisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
'' SIG '' AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIM/PbBYqwqbi
'' SIG '' lA3rG2vVI/jiEGyvy8AcVmNxe6m7zfw9MDAGCisGAQQB
'' SIG '' gjcCAQwxIjAgoB6AHABBAGQAbwBiAGUAIABJAG4ARABl
'' SIG '' AHMAaQBnAG4wDQYJKoZIhvcNAQEBBQAEggEALymcGJqW
'' SIG '' lEsDSggHRD1KytxSYYTPpdwcsDUCsHW1/yTQDvRwqKYX
'' SIG '' W1mQ8JuVB+uw9haAQ1ya5MKPrOUM9UQ0tUx+PmvxinhF
'' SIG '' gvujyM03hGj5oJG4X2Xp9O+4+BTQyKxlNWnucj3vtXdj
'' SIG '' JTAVmnejmwTxl/Xp11BMaNg7FwMScvHnxyJJJ3jkw37/
'' SIG '' eIwzZl8tPLh230iTsNd/WITM7HFIqhOXImfHgKyX85op
'' SIG '' ttdIGjGg4E9fyOhi1BtgaKMXM30zXRhPmZLyRF/Wfzc7
'' SIG '' X2Njbhq5zMkvoHA8vMS0PKcPJM5gzv3Qog6RSVIAWr4D
'' SIG '' mQSigeMciySM1trh+59cXR4zRqGCEbIwghGuBgorBgEE
'' SIG '' AYI3AwMBMYIRnjCCEZoGCSqGSIb3DQEHAqCCEYswghGH
'' SIG '' AgEDMQ8wDQYJYIZIAWUDBAIBBQAwdwYLKoZIhvcNAQkQ
'' SIG '' AQSgaARmMGQCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFl
'' SIG '' AwQCAQUABCCpd60yc0A/MBDpPgDQsqhFAW6iXIIJzPXd
'' SIG '' XDQyKV8xdQIQavoWQGjiN9hHwDbmZDKaYRgPMjAyMjA0
'' SIG '' MTgxOTAyMjBaoIINfDCCBsYwggSuoAMCAQICEAp6Soie
'' SIG '' yZlCkAZjOE2Gl50wDQYJKoZIhvcNAQELBQAwYzELMAkG
'' SIG '' A1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
'' SIG '' MTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJT
'' SIG '' QTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQTAeFw0y
'' SIG '' MjAzMjkwMDAwMDBaFw0zMzAzMTQyMzU5NTlaMEwxCzAJ
'' SIG '' BgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
'' SIG '' LjEkMCIGA1UEAxMbRGlnaUNlcnQgVGltZXN0YW1wIDIw
'' SIG '' MjIgLSAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
'' SIG '' CgKCAgEAuSqWI6ZcvF/WSfAVghj0M+7MXGzj4CUu0jHk
'' SIG '' PECu+6vE43hdflw26vUljUOjges4Y/k8iGnePNIwUQ0x
'' SIG '' B7pGbumjS0joiUF/DbLW+YTxmD4LvwqEEnFsoWImAdPO
'' SIG '' w2z9rDt+3Cocqb0wxhbY2rzrsvGD0Z/NCcW5QWpFQiNB
'' SIG '' Wvhg02UsPn5evZan8Pyx9PQoz0J5HzvHkwdoaOVENFJf
'' SIG '' D1De1FksRHTAMkcZW+KYLo/Qyj//xmfPPJOVToTpdhiY
'' SIG '' mREUxSsMoDPbTSSF6IKU4S8D7n+FAsmG4dUYFLcERfPg
'' SIG '' OL2ivXpxmOwV5/0u7NKbAIqsHY07gGj+0FmYJs7g7a5/
'' SIG '' KC7CnuALS8gI0TK7g/ojPNn/0oy790Mj3+fDWgVifnAs
'' SIG '' 5SuyPWPqyK6BIGtDich+X7Aa3Rm9n3RBCq+5jgnTdKEv
'' SIG '' sFR2wZBPlOyGYf/bES+SAzDOMLeLD11Es0MdI1DNkdcv
'' SIG '' nfv8zbHBp8QOxO9APhk6AtQxqWmgSfl14ZvoaORqDI/r
'' SIG '' 5LEhe4ZnWH5/H+gr5BSyFtaBocraMJBr7m91wLA2JrII
'' SIG '' O/+9vn9sExjfxm2keUmti39hhwVo99Rw40KV6J67m0uy
'' SIG '' 4rZBPeevpxooya1hsKBBGBlO7UebYZXtPgthWuo+epiS
'' SIG '' Uc0/yUTngIspQnL3ebLdhOon7v59emsCAwEAAaOCAYsw
'' SIG '' ggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAA
'' SIG '' MBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZ
'' SIG '' MBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSME
'' SIG '' GDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4E
'' SIG '' FgQUjWS3iSH+VlhEhGGn6m8cNo/drw0wWgYDVR0fBFMw
'' SIG '' UTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29t
'' SIG '' L0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRp
'' SIG '' bWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMw
'' SIG '' gYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2lj
'' SIG '' ZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2Vy
'' SIG '' dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0
'' SIG '' UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDAN
'' SIG '' BgkqhkiG9w0BAQsFAAOCAgEADS0jdKbR9fjqS5k/AeT2
'' SIG '' DOSvFp3Zs4yXgimcQ28BLas4tXARv4QZiz9d5YZPvpM6
'' SIG '' 3io5WjlO2IRZpbwbmKrobO/RSGkZOFvPiTkdcHDZTt8j
'' SIG '' ImzV3/ZZy6HC6kx2yqHcoSuWuJtVqRprfdH1AglPgtal
'' SIG '' c4jEmIDf7kmVt7PMxafuDuHvHjiKn+8RyTFKWLbfOHzL
'' SIG '' +lz35FO/bgp8ftfemNUpZYkPopzAZfQBImXH6l50pls1
'' SIG '' klB89Bemh2RPPkaJFmMga8vye9A140pwSKm25x1gvQQi
'' SIG '' FSVwBnKpRDtpRxHT7unHoD5PELkwNuTzqmkJqIt+ZKJl
'' SIG '' lBH7bjLx9bs4rc3AkxHVMnhKSzcqTPNc3LaFwLtwMFV4
'' SIG '' 1pj+VG1/calIGnjdRncuG3rAM4r4SiiMEqhzzy350yPy
'' SIG '' nhngDZQooOvbGlGglYKOKGukzp123qlzqkhqWUOuX+r4
'' SIG '' DwZCnd8GaJb+KqB0W2Nm3mssuHiqTXBt8CzxBxV+NbTm
'' SIG '' tQyimaXXFWs1DoXW4CzM4AwkuHxSCx6ZfO/IyMWMWGmv
'' SIG '' qz3hz8x9Fa4Uv4px38qXsdhH6hyF4EVOEhwUKVjMb9N/
'' SIG '' y77BDkpvIJyu2XMyWQjnLZKhGhH+MpimXSuX4IvTnMxt
'' SIG '' tQ2uR2M4RxdbbxPaahBuH0m3RFu0CAqHWlkEdhGhp3cC
'' SIG '' ExwwggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5b
'' SIG '' MA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUw
'' SIG '' EwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
'' SIG '' dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0
'' SIG '' IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBa
'' SIG '' Fw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
'' SIG '' FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMy
'' SIG '' RGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1
'' SIG '' NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEB
'' SIG '' AQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE
'' SIG '' 8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6
'' SIG '' ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOx
'' SIG '' s+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt
'' SIG '' 69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rw
'' SIG '' N3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y
'' SIG '' 3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLF
'' SIG '' uk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yem
'' SIG '' j052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae
'' SIG '' 5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0
'' SIG '' QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/w
'' SIG '' ojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxm
'' SIG '' SVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
'' SIG '' +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjB
'' SIG '' Jgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJg
'' SIG '' o1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TN
'' SIG '' sQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIB
'' SIG '' ADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8w
'' SIG '' HwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08w
'' SIG '' DgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUF
'' SIG '' BwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYY
'' SIG '' aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUF
'' SIG '' BzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20v
'' SIG '' RGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8E
'' SIG '' PDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
'' SIG '' b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNV
'' SIG '' HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJ
'' SIG '' KoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tgh
'' SIG '' QuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
'' SIG '' B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiC
'' SIG '' qBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1D
'' SIG '' X+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8
'' SIG '' dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6Z
'' SIG '' JxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/um
'' SIG '' nXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzp
'' SIG '' SwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNB
'' SIG '' zU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YB
'' SIG '' T70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f5
'' SIG '' 6PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZ
'' SIG '' ynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3
'' SIG '' OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADN
'' SIG '' XcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2
'' SIG '' VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8
'' SIG '' MYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUG
'' SIG '' A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRp
'' SIG '' Z2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYg
'' SIG '' VGltZVN0YW1waW5nIENBAhAKekqInsmZQpAGYzhNhped
'' SIG '' MA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzEN
'' SIG '' BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjIw
'' SIG '' NDE4MTkwMjIwWjArBgsqhkiG9w0BCRACDDEcMBowGDAW
'' SIG '' BBSFCPOGUVyz0wd9trS3wH8bSl5B3jAvBgkqhkiG9w0B
'' SIG '' CQQxIgQgFZAEb8g+3j2Kg+20Idt0vKhxx5y2+XPmgT+E
'' SIG '' PG3Z4BcwNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgnaaQ
'' SIG '' FcNJxsGJeEW6NYKtcMiPpCk722q+nCvSU5J55jswDQYJ
'' SIG '' KoZIhvcNAQEBBQAEggIAlFjLNaQWLgNqkoQhu8n8ZvDC
'' SIG '' bS6OQNGfqGUeLpUVMopfJkIzlScMAX8zW1lq/WBc05Bw
'' SIG '' jyi1JHLOLQdSK4bL0zLHPcF7+z61wikVjVSRpT6bdBjO
'' SIG '' J27KsrsWUQ9+BRoTg2n3VDt5wW5eY+NMptvJAd4UDY1d
'' SIG '' 11XjTwzTXjWLHoqsykgxbWrleBGJQk902VuoJ+dQ8QFF
'' SIG '' IZbFOaDLB0KCiB3T6fRRwP5XYgjrNSnzBHmoAgSa4eVj
'' SIG '' 2p/jNvFsI1qdNL/5HYDLKk8whehPuM/wY59s8ROuwrO+
'' SIG '' OKEMNdGyvvfASEk/WqFcoxkAn89pzHBxs2GNZEHo7o+L
'' SIG '' yoYed5Ij7tIatJMrt+h3Lp80JjS7Gsi6gbIIOVlyFnYe
'' SIG '' YEVVFiqC8S2xU5SNX6lfRZzvSLRSY7q/Xhhmbny14FO4
'' SIG '' NoR1dFwyqGwqmXqGlZZwAQoglEkWCdHxsJ1l4+N6IMOJ
'' SIG '' Oc0q0KCrNjFKE4TFBLnM2vo7UxJNwyotny2rG4gXvWuP
'' SIG '' wFDqcOiih6nnN1FCtxnx2vGKVPpX30olOS/NDEJdraUG
'' SIG '' hjGwDqJ8GASUDQIZlvEvq1DElyZsQHbZdb2FaAwGeslm
'' SIG '' PazXWPySN3zqnDl3gB40OgqMa3tXjZNuUeF8n980GyEE
'' SIG '' ZkVaZfV/5geUGPrd3gt2RYl0EKQ6K2zjy8MeFFfH148=
'' SIG '' End signature block
